//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Collections.Generic;
using System.Configuration;
using System.Reflection;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;
using System.Globalization;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents a strategy that will transform a nested element into a dictionary of types.
	/// </summary>
    public class DictionaryFactory<TObjectKey, TObjectToCreate, TObjectConfiguration> : ObjectFactoryBase<IDictionary<TObjectKey, TObjectToCreate>, IEnumerable<TObjectConfiguration>>
	{
		private readonly Type dictionaryType;
		private readonly string keyPropertyName;
        private readonly IObjectFactory<TObjectToCreate, TObjectConfiguration> elementFactory;

		/// <summary>
		/// Initialize a new instance of the <see cref="DictionaryFactory"/> class with the type of dictionary to create, the property name that is the key to the new dictionary, and a transformation strategy for the nested element.
		/// </summary>
		/// <param name="dictionaryType">The type of dictionary to create.</param>
		/// <param name="keyPropertyName">The property of the nested element that is the key for the dictionary.</param>
		/// <param name="elementTransformationStrategy">The <see cref="TransformationStrategy"/> for the nested element.</param>
        public DictionaryFactory(Type dictionaryType, string keyPropertyName, IObjectFactory<TObjectToCreate, TObjectConfiguration> elementFactory)
		{
			if (dictionaryType == null) throw new ArgumentNullException("dictionaryType");
			if (keyPropertyName == null) throw new ArgumentNullException("keyPropertyName");
            if (elementFactory == null) throw new ArgumentNullException("elementFactory");
			if (!typeof(IDictionary<TObjectKey, TObjectToCreate>).IsAssignableFrom(dictionaryType))
			{
				throw new ArgumentException(Resources.ExceptionTypeNotIDictionary, "dictionaryType");
			}

			this.dictionaryType = dictionaryType;
			this.keyPropertyName = keyPropertyName;
            this.elementFactory = elementFactory;
		}

		/// <summary>
		/// Gets the type of dictionary to create.
		/// </summary>
		/// <value>
		/// The type of dictionary to create.
		/// </value>
		public Type DictionaryType
		{
			get { return dictionaryType; }
		}

		/// <summary>
		/// Gets the property name of the nested element that will be the key for the dictioanry.
		/// </summary>
		/// <value>
		/// The property name of the nested element that will be the key for the dictioanry.
		/// </value>
		public string KeyPropertyName
		{
			get { return keyPropertyName; }
		}

		/// <summary>
		/// Gets the <see cref="TransformationStrategy"/> used for the nested element.
		/// </summary>
		/// <value>
		/// The <see cref="TransformationStrategy"/> used for the nested element.
		/// </value>
        public IObjectFactory<TObjectToCreate, TObjectConfiguration> ElementFactory
		{
            get { return elementFactory; }
        }

        public override IDictionary<TObjectKey, TObjectToCreate> Create(IEnumerable<TObjectConfiguration> configurations, IConfigurationSource configurationSource)
        {
            IDictionary<TObjectKey, TObjectToCreate> result = null;

            if (configurations != null)
            {
                //IEnumerable<TObjectConfiguration> configurations = sourceValue as IEnumerable<TObjectConfiguration>;
                //if (configurations == null)
                //{
                //    throw new ArgumentException(Resources.ExceptionSourceNotICollection, "sourceValue");
                //}

                result = (IDictionary<TObjectKey, TObjectToCreate>)Activator.CreateInstance(dictionaryType);
                foreach (TObjectConfiguration objectConfiguration in configurations)
                {
                    TObjectKey val = (TObjectKey)objectConfiguration.GetType().InvokeMember(
                        keyPropertyName,
                        BindingFlags.GetProperty | BindingFlags.Public | BindingFlags.Instance,
                        null,
                        objectConfiguration,
                        null,
                        CultureInfo.CurrentCulture);
                    result.Add(val,
                        elementFactory.Create(objectConfiguration, configurationSource));
                }
            }

            return result;
        }
    }
}
